
// Copyright (c) 2010, Peter Barrett
/*
** Permission to use, copy, modify, and/or distribute this software for  
** any purpose with or without fee is hereby granted, provided that the  
** above copyright notice and this permission notice appear in all copies.  
**  
** THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL  
** WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED  
** WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR  
** BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES  
** OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,  
** WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,  
** ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS  
** SOFTWARE.  
*/

#ifndef __USBCORE_H__
#define __USBCORE_H__

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <stdint.h>

class USBDevice_
{
 public:
  USBDevice_();
  bool configured();

  void attach();
  void detach(); // Serial port goes down too...
  void poll();
  bool wakeupHost(); // returns false, when wakeup cannot be processed
};

extern USBDevice_ USBDevice;

// Standard requests
#define GET_STATUS 0
#define CLEAR_FEATURE 1
#define SET_FEATURE 3
#define SET_ADDRESS 5
#define GET_DESCRIPTOR 6
#define SET_DESCRIPTOR 7
#define GET_CONFIGURATION 8
#define SET_CONFIGURATION 9
#define GET_INTERFACE 10
#define SET_INTERFACE 11

// bmRequestType
#define REQUEST_HOSTTODEVICE 0x00
#define REQUEST_DEVICETOHOST 0x80
#define REQUEST_DIRECTION 0x80

#define REQUEST_STANDARD 0x00
#define REQUEST_CLASS 0x20
#define REQUEST_VENDOR 0x40
#define REQUEST_TYPE 0x60

#define REQUEST_DEVICE 0x00
#define REQUEST_INTERFACE 0x01
#define REQUEST_ENDPOINT 0x02
#define REQUEST_OTHER 0x03
#define REQUEST_RECIPIENT 0x03

#define REQUEST_DEVICETOHOST_CLASS_INTERFACE (REQUEST_DEVICETOHOST | REQUEST_CLASS | REQUEST_INTERFACE)
#define REQUEST_HOSTTODEVICE_CLASS_INTERFACE (REQUEST_HOSTTODEVICE | REQUEST_CLASS | REQUEST_INTERFACE)
#define REQUEST_DEVICETOHOST_STANDARD_INTERFACE (REQUEST_DEVICETOHOST | REQUEST_STANDARD | REQUEST_INTERFACE)

// Class requests

#define CDC_SET_LINE_CODING 0x20
#define CDC_GET_LINE_CODING 0x21
#define CDC_SET_CONTROL_LINE_STATE 0x22
#define CDC_SEND_BREAK 0x23

#define MSC_RESET 0xFF
#define MSC_GET_MAX_LUN 0xFE

// Descriptors

#define USB_DEVICE_DESC_SIZE 18
#define USB_CONFIGUARTION_DESC_SIZE 9
#define USB_INTERFACE_DESC_SIZE 9
#define USB_ENDPOINT_DESC_SIZE 7

#define USB_DEVICE_DESCRIPTOR_TYPE 1
#define USB_CONFIGURATION_DESCRIPTOR_TYPE 2
#define USB_STRING_DESCRIPTOR_TYPE 3
#define USB_INTERFACE_DESCRIPTOR_TYPE 4
#define USB_ENDPOINT_DESCRIPTOR_TYPE 5

// usb_20.pdf Table 9.6 Standard Feature Selectors
#define DEVICE_REMOTE_WAKEUP 1
#define ENDPOINT_HALT 2
#define TEST_MODE 3

// usb_20.pdf Figure 9-4. Information Returned by a GetStatus() Request to a Device
#define FEATURE_SELFPOWERED_ENABLED (1 << 0)
#define FEATURE_REMOTE_WAKEUP_ENABLED (1 << 1)

#define USB_DEVICE_CLASS_COMMUNICATIONS 0x02
#define USB_DEVICE_CLASS_HUMAN_INTERFACE 0x03
#define USB_DEVICE_CLASS_STORAGE 0x08
#define USB_DEVICE_CLASS_VENDOR_SPECIFIC 0xFF

#define USB_CONFIG_POWERED_MASK 0x40
#define USB_CONFIG_BUS_POWERED 0x80
#define USB_CONFIG_SELF_POWERED 0xC0
#define USB_CONFIG_REMOTE_WAKEUP 0x20

// bMaxPower in Configuration Descriptor
#define USB_CONFIG_POWER_MA(mA) ((mA) / 2)

// bEndpointAddress in Endpoint Descriptor
#define USB_ENDPOINT_DIRECTION_MASK 0x80
#define USB_ENDPOINT_OUT(addr) (lowByte((addr) | 0x00))
#define USB_ENDPOINT_IN(addr) (lowByte((addr) | 0x80))

#define USB_ENDPOINT_TYPE_MASK 0x03
#define USB_ENDPOINT_TYPE_CONTROL 0x00
#define USB_ENDPOINT_TYPE_ISOCHRONOUS 0x01
#define USB_ENDPOINT_TYPE_BULK 0x02
#define USB_ENDPOINT_TYPE_INTERRUPT 0x03

#define TOBYTES(x) ((x)&0xFF), (((x) >> 8) & 0xFF)

#define CDC_V1_10 0x0110
#define CDC_COMMUNICATION_INTERFACE_CLASS 0x02

#define CDC_CALL_MANAGEMENT 0x01
#define CDC_ABSTRACT_CONTROL_MODEL 0x02
#define CDC_HEADER 0x00
#define CDC_ABSTRACT_CONTROL_MANAGEMENT 0x02
#define CDC_UNION 0x06
#define CDC_CS_INTERFACE 0x24
#define CDC_CS_ENDPOINT 0x25
#define CDC_DATA_INTERFACE_CLASS 0x0A

#define MSC_SUBCLASS_SCSI 0x06
#define MSC_PROTOCOL_BULK_ONLY 0x50

#ifndef USB_VERSION
#define USB_VERSION 0x200
#endif

#define TRANSFER_PGM 0x80
#define TRANSFER_RELEASE 0x40
#define TRANSFER_ZERO 0x20

// Device
typedef struct
{
  uint8_t len;         // 18
  uint8_t dtype;       // 1 USB_DEVICE_DESCRIPTOR_TYPE
  uint16_t usbVersion; // 0x200 or 0x210
  uint8_t deviceClass;
  uint8_t deviceSubClass;
  uint8_t deviceProtocol;
  uint8_t packetSize0; // Packet 0
  uint16_t idVendor;
  uint16_t idProduct;
  uint16_t deviceVersion; // 0x100
  uint8_t iManufacturer;
  uint8_t iProduct;
  uint8_t iSerialNumber;
  uint8_t bNumConfigurations;
} DeviceDescriptor;

// Config
typedef struct
{
  uint8_t len;   // 9
  uint8_t dtype; // 2
  uint16_t clen; // total length
  uint8_t numInterfaces;
  uint8_t config;
  uint8_t iconfig;
  uint8_t attributes;
  uint8_t maxPower;
} ConfigDescriptor;

// String

// Interface
typedef struct
{
  uint8_t len;   // 9
  uint8_t dtype; // 4
  uint8_t number;
  uint8_t alternate;
  uint8_t numEndpoints;
  uint8_t interfaceClass;
  uint8_t interfaceSubClass;
  uint8_t protocol;
  uint8_t iInterface;
} InterfaceDescriptor;

// Endpoint
typedef struct
{
  uint8_t len;   // 7
  uint8_t dtype; // 5
  uint8_t addr;
  uint8_t attr;
  uint16_t packetSize;
  uint8_t interval;
} EndpointDescriptor;

// Interface Association Descriptor
// Used to bind 2 interfaces together in CDC compostite device
typedef struct
{
  uint8_t len;   // 8
  uint8_t dtype; // 11
  uint8_t firstInterface;
  uint8_t interfaceCount;
  uint8_t functionClass;
  uint8_t funtionSubClass;
  uint8_t functionProtocol;
  uint8_t iInterface;
} IADDescriptor;

// CDC CS interface descriptor
typedef struct
{
  uint8_t len;   // 5
  uint8_t dtype; // 0x24
  uint8_t subtype;
  uint8_t d0;
  uint8_t d1;
} CDCCSInterfaceDescriptor;

typedef struct
{
  uint8_t len;   // 4
  uint8_t dtype; // 0x24
  uint8_t subtype;
  uint8_t d0;
} CDCCSInterfaceDescriptor4;

typedef struct
{
  uint8_t len;
  uint8_t dtype;   // 0x24
  uint8_t subtype; // 1
  uint8_t bmCapabilities;
  uint8_t bDataInterface;
} CMFunctionalDescriptor;

typedef struct
{
  uint8_t len;
  uint8_t dtype;   // 0x24
  uint8_t subtype; // 1
  uint8_t bmCapabilities;
} ACMFunctionalDescriptor;

typedef struct
{
  // IAD
  IADDescriptor iad; // Only needed on compound device

  // Control
  InterfaceDescriptor cif; //
  CDCCSInterfaceDescriptor header;
  CMFunctionalDescriptor callManagement;         // Call Management
  ACMFunctionalDescriptor controlManagement;     // ACM
  CDCCSInterfaceDescriptor functionalDescriptor; // CDC_UNION
  EndpointDescriptor cifin;

  // Data
  InterfaceDescriptor dif;
  EndpointDescriptor in;
  EndpointDescriptor out;
} CDCDescriptor;

typedef struct
{
  InterfaceDescriptor msc;
  EndpointDescriptor in;
  EndpointDescriptor out;
} MSCDescriptor;

#define D_DEVICE(_class, _subClass, _proto, _packetSize0, _vid, _pid, _version, _im, _ip, _is, _configs)       \
  {                                                                                                            \
    18, 1, USB_VERSION, _class, _subClass, _proto, _packetSize0, _vid, _pid, _version, _im, _ip, _is, _configs \
  }

#define D_CONFIG(_totalLength, _interfaces)                                                                            \
  {                                                                                                                    \
    9, 2, _totalLength, _interfaces, 1, 0, USB_CONFIG_BUS_POWERED | USB_CONFIG_REMOTE_WAKEUP, USB_CONFIG_POWER_MA(500) \
  }

#define D_INTERFACE(_n, _numEndpoints, _class, _subClass, _protocol) \
  {                                                                  \
    9, 4, _n, 0, _numEndpoints, _class, _subClass, _protocol, 0      \
  }

#define D_ENDPOINT(_addr, _attr, _packetSize, _interval) \
  {                                                      \
    7, 5, _addr, _attr, _packetSize, _interval           \
  }

#define D_IAD(_firstInterface, _count, _class, _subClass, _protocol) \
  {                                                                  \
    8, 11, _firstInterface, _count, _class, _subClass, _protocol, 0  \
  }

#define D_CDCCS(_subtype, _d0, _d1) \
  {                                 \
    5, 0x24, _subtype, _d0, _d1     \
  }
#define D_CDCCS4(_subtype, _d0) \
  {                             \
    4, 0x24, _subtype, _d0      \
  }

// Bootloader related fields
// Old Caterina bootloader places the MAGIC key into unsafe RAM locations (it can be rewritten
// by the running sketch before to actual reboot).
// Newer bootloaders, recognizable by the LUFA "signature" at the end of the flash, can handle both
// the usafe and the safe location.
#ifndef MAGIC_KEY
#define MAGIC_KEY 0x7777
#endif

#ifndef MAGIC_KEY_POS
#define MAGIC_KEY_POS 0x0800
#endif

#ifndef NEW_LUFA_SIGNATURE
#define NEW_LUFA_SIGNATURE 0xDCFB
#endif

/* This core is PluggableUSB capable */
#define PLUGGABLE_USB_ENABLED

/* Endpoints number */
#if defined(EPRST6)
#define USB_ENDPOINTS 7 // AtMegaxxU4
#else
#define USB_ENDPOINTS 5 // AtMegaxxU2
#endif

#define EP_TYPE_CONTROL (0x00)
#define EP_TYPE_BULK_IN ((1 << EPTYPE1) | (1 << EPDIR))
#define EP_TYPE_BULK_OUT (1 << EPTYPE1)
#define EP_TYPE_INTERRUPT_IN ((1 << EPTYPE1) | (1 << EPTYPE0) | (1 << EPDIR))
#define EP_TYPE_INTERRUPT_OUT ((1 << EPTYPE1) | (1 << EPTYPE0))
#define EP_TYPE_ISOCHRONOUS_IN ((1 << EPTYPE0) | (1 << EPDIR))
#define EP_TYPE_ISOCHRONOUS_OUT (1 << EPTYPE0)

// This definitions is usefull if you want to reduce the EP_SIZE to 16
// at the moment only 64 and 16 as EP_SIZE for all EPs are supported except the control endpoint
#ifndef USB_EP_SIZE
#define USB_EP_SIZE 64
#endif

#define ISERIAL_MAX_LEN 20

#define CDC_INTERFACE_COUNT 2
#define CDC_ENPOINT_COUNT 3

#define CDC_ACM_INTERFACE 0  // CDC ACM
#define CDC_DATA_INTERFACE 1 // CDC Data
#define CDC_FIRST_ENDPOINT 1
#define CDC_ENDPOINT_ACM (CDC_FIRST_ENDPOINT) // CDC First
#define CDC_ENDPOINT_OUT (CDC_FIRST_ENDPOINT + 1)
#define CDC_ENDPOINT_IN (CDC_FIRST_ENDPOINT + 2)

#define INTERFACE_COUNT (MSC_INTERFACE + MSC_INTERFACE_COUNT)

#define CDC_RX CDC_ENDPOINT_OUT
#define CDC_TX CDC_ENDPOINT_IN

#define IMANUFACTURER 1
#define IPRODUCT 2
#define ISERIAL 3

#endif
